extends Node

# ! Comments prefixed with "!" mean they are extra info. Comments without them
# ! should be kept because they give your mod structure and make it easier to
# ! read by other modders
# ! Comments with "?" should be replaced by you with the appropriate information

# ! This template file is statically typed. You don't have to do that, but it can help avoid bugs
# ! You can learn more about static typing in the docs
# ! https://docs.godotengine.org/en/3.5/tutorials/scripting/gdscript/static_typing.html

# ? Brief overview of what your mod does...

const MOD_DIR := "CST1229-BALauncherModLoader" # Name of the directory that this file is in
const LOG_NAME := "CST1229-BALauncherModLoader:Main" # Full ID of the mod (AuthorName-ModName)

var mod_dir_path := ""
var extensions_dir_path := ""
var translations_dir_path := ""

var lite_mode := OS.has_feature("mobile")
var no_launcher_updates := OS.has_feature("mobile")

var use_file_browser = ["Android", "iOS", "Web"].has(OS.get_name());

# ! your _ready func.
func _init() -> void:
	ModLoaderLog.info("Init", LOG_NAME)
	mod_dir_path = ModLoaderMod.get_unpacked_dir().path_join(MOD_DIR)
	
	# for identifying mod loader screenshots
	add_child(load("res://mods-unpacked/CST1229-BALauncherModLoader/IdentifyingSpeck.tscn").instantiate())
	
	set_mod_types();
	
	# Add extensions
	install_script_extensions()
	install_script_hook_files()


func install_script_extensions() -> void:
	# ! any script extensions should go in this directory, and should follow the same directory structure as vanilla
	extensions_dir_path = mod_dir_path.path_join("extensions")
	ModLoaderMod.install_script_extension(extensions_dir_path.path_join("launcher.gd"))


func install_script_hook_files() -> void:
	extensions_dir_path = mod_dir_path.path_join("extensions")

func start_init_mods():
	var ml := $/root/ModLoader;
	var root := $/root;
	root.remove_child(ml);
	ml.request_ready();
	request_ready();
	init_mods_late();
	root.get_tree().process_frame.connect(func():
		root.add_child(ml);
	, CONNECT_ONE_SHOT);

var restart_warned := false;
var mod_types := {};
func set_mod_types() -> void:
	var path: String = ModLoaderStore.ml_options.override_path_to_mods;
	if path == "":
		path = "mods";
	for mod in ModLoaderStore.mod_data.values():
		if mod not in mod_types:
			mod_types[mod] = path;

# literally just mostly copy-pasted from the mod loader (with changes noted in comments)
var late_loaded_mods := {}
func reload_game_mods():
	$"/root/ModLoader/CST1229-BALauncherModLoader".remove_lateloaded_mods();
	
	ModLoaderStore.ml_options.override_path_to_mods = "game-mods" if !use_file_browser else "user://game-mods";
	if !OS.has_feature("editor"):
		DirAccess.make_dir_recursive_absolute(ModLoaderStore.ml_options.override_path_to_mods);
	ModLoaderStore.ml_options.load_from_unpacked = false;
	$"/root/ModLoader/CST1229-BALauncherModLoader".load_mods_late();
func remove_lateloaded_mods():
	# start CST added
	for mod in late_loaded_mods.duplicate():
		remove_mod(mod);
	# end CST added
func load_mods_late():
	# --- Start loading mods ---
	var loaded_count := 0

	# mod_path can be a directory in mods-unpacked or a mod.zip
	var mod_paths := _ModLoaderPath.get_mod_paths_from_all_sources()

	ModLoaderLog.debug("Found %s mods at the following paths:\n\t - %s" % [mod_paths.size(), "\n\t - ".join(mod_paths)], ModLoader.LOG_NAME)

	for mod_path in mod_paths:
		var is_zip := _ModLoaderPath.is_zip(mod_path)
		# CST: added
		var dir_name
		if is_zip:
			dir_name = _ModLoaderFile.get_mod_dir_name_in_zip(mod_path)
		else:
			dir_name = mod_path.split("/")[-1]
		if dir_name in ModLoaderStore.mod_data: continue
		# CST: end added

		# Load manifest file
		var manifest_data: Dictionary = _ModLoaderFile.load_manifest_file(mod_path)

		var manifest := ModManifest.new(manifest_data, mod_path)

		if not manifest.validation_messages_error.is_empty():
			ModLoaderLog.error(
				"The mod from path \"%s\" cannot be loaded. Manifest validation failed with the following errors:\n\t - %s" %
				[mod_path, "\n\t - ".join(manifest.validation_messages_error)], ModLoader.LOG_NAME
			)

		# Init ModData
		var mod := ModData.new(manifest, mod_path)

		if not mod.load_errors.is_empty():
			ModLoaderStore.ml_options.disabled_mods.append(mod.manifest.get_mod_id())
			ModLoaderLog.error(
				"The mod from path \"%s\" cannot be loaded. ModData initialization has failed with the following errors:\n\t - %s" %
				[mod_path, "\n\t - ".join(mod.load_errors)], ModLoader.LOG_NAME
			)

		# Using mod.dir_name here allows us to store the ModData even if manifest validation fails.
		ModLoaderStore.mod_data[mod.dir_name] = mod
		# CST: added
		late_loaded_mods[mod] = true
		# CST: end added

		if mod.is_loadable:
			if is_zip:
				var is_mod_loaded_successfully := ProjectSettings.load_resource_pack(mod_path, false)

				if not is_mod_loaded_successfully:
					ModLoaderLog.error("Failed to load mod zip from path \"%s\" into the virtual filesystem." % mod_path, ModLoader.LOG_NAME)
					continue

				# Notifies developer of an issue with Godot, where using `load_resource_pack`
				# in the editor WIPES the entire virtual res:// directory the first time you
				# use it. This means that unpacked mods are no longer accessible, because they
				# no longer exist in the file system. So this warning basically says
				# "don't use ZIPs with unpacked mods!"
				# https://github.com/godotengine/godot/issues/19815
				# https://github.com/godotengine/godot/issues/16798
				if ModLoaderStore.has_feature.editor:
					ModLoaderLog.hint(
						"Loading any resource packs (.zip/.pck) with `load_resource_pack` will WIPE the entire virtual res:// directory. " +
						"If you have any unpacked mods in %s, they will not be loaded.Please unpack your mod ZIPs instead, and add them to %s" %
						[_ModLoaderPath.get_unpacked_mods_dir_path(), _ModLoaderPath.get_unpacked_mods_dir_path()], ModLoader.LOG_NAME, true
					)

			ModLoaderLog.success("%s loaded." % mod_path, ModLoader.LOG_NAME)
			loaded_count += 1

	ModLoaderLog.success("DONE: Loaded %s mod files into the virtual filesystem" % loaded_count, ModLoader.LOG_NAME)

	# Update the mod_list for each user profile
	var _success_update_mod_lists := ModLoaderUserProfile._update_mod_lists()

	# Update active state of mods based on the current user profile
	ModLoaderUserProfile._update_disabled_mods()
	
	# Load all Mod Configs
	for dir_name in ModLoaderStore.mod_data:
		var mod: ModData = ModLoaderStore.mod_data[dir_name]
		if not mod.is_loadable:
			continue
		if mod.manifest.get("config_schema") and not mod.manifest.config_schema.is_empty():
			mod.load_configs()
	ModLoaderLog.success("DONE: Loaded all mod configs", ModLoader.LOG_NAME)

	# Check for mods with load_before. If a mod is listed in load_before,
	# add the current mod to the dependencies of the the mod specified
	# in load_before.
	for dir_name in ModLoaderStore.mod_data:
		var mod: ModData = ModLoaderStore.mod_data[dir_name]
		if not mod.is_loadable:
			continue
		_ModLoaderDependency.check_load_before(mod)

	# Run optional dependency checks.
	# If a mod depends on another mod that hasn't been loaded,
	# the dependent mod will be loaded regardless.
	for dir_name in ModLoaderStore.mod_data:
		var mod: ModData = ModLoaderStore.mod_data[dir_name]
		if not mod.is_loadable:
			continue
		var _is_circular := _ModLoaderDependency.check_dependencies(mod, false)

	# Run dependency checks. If a mod depends on another
	# mod that hasn't been loaded, the dependent mod won't be loaded.
	for dir_name in ModLoaderStore.mod_data:
		var mod: ModData = ModLoaderStore.mod_data[dir_name]
		if not mod.is_loadable:
			continue
		var _is_circular := _ModLoaderDependency.check_dependencies(mod)
	set_mod_types();

func init_mods_late():
	# Update the mod_list for each user profile
	var _success_update_mod_lists := ModLoaderUserProfile._update_mod_lists()

	# Update active state of mods based on the current user profile
	ModLoaderUserProfile._update_disabled_mods()
	
	# Load all Mod Configs
	for dir_name in ModLoaderStore.mod_data:
		var mod: ModData = ModLoaderStore.mod_data[dir_name]
		if not mod.is_loadable:
			continue
		if mod.manifest.get("config_schema") and not mod.manifest.config_schema.is_empty():
			mod.load_configs()
	ModLoaderLog.success("DONE: Loaded all mod configs", ModLoader.LOG_NAME)
	
	# Sort mod_load_order by the importance score of the mod
	ModLoaderStore.mod_load_order = _ModLoaderDependency.get_load_order(ModLoaderStore.mod_data.values())

	# Log mod order
	for mod_index in ModLoaderStore.mod_load_order.size():
		var mod: ModData = ModLoaderStore.mod_load_order[mod_index]
		ModLoaderLog.info("mod_load_order -> %s) %s" % [mod_index + 1, mod.dir_name], ModLoader.LOG_NAME)

	# Instance every mod and add it as a node to the Mod Loader
	for mod in ModLoaderStore.mod_load_order:
		mod = mod as ModData
		
		# CST added
		if mod not in late_loaded_mods: continue
		# end CST added

		# Continue if mod is disabled
		if not mod.is_active or not mod.is_loadable:
			continue

		ModLoaderLog.info("Initializing -> %s" % mod.manifest.get_mod_id(), ModLoader.LOG_NAME)
		ModLoader._init_mod(mod)

	ModLoaderLog.debug_json_print("mod data", ModLoaderStore.mod_data, ModLoader.LOG_NAME)

	ModLoaderLog.success("DONE: Completely finished loading mods", ModLoader.LOG_NAME)

	_ModLoaderSceneExtension.refresh_scenes()

	_ModLoaderSceneExtension.handle_scene_extensions()

func remove_mod(mod: ModData) -> void:
	ModLoaderStore.mod_data.erase(ModLoaderStore.mod_data.find_key(mod));
	var manifest := mod.manifest;
	var id := manifest.get_mod_id();
	ModLoaderStore.ml_options.disabled_mods.erase(id);
	late_loaded_mods.erase(mod);
	
