extends CanvasLayer

@onready var mods_box: VBoxContainer = $ModsScroll/MarginContainer/ModsBox;

@onready var restart_button: Button = $ModsScroll/MarginContainer/ModsBox/HBoxContainer/Restart

@onready var click: AudioStreamPlayer = $Click;

const MOD = preload("res://mods-unpacked/CST1229-BALauncherModLoader/scenes/Mod.tscn");
var mod_nodes: Array[Node] = [];

var changed_mod_config: Dictionary = {};

@onready var mod_main = $"/root/ModLoader/CST1229-BALauncherModLoader";

var use_file_browser = ["Android", "iOS", "Web"].has(OS.get_name());
@onready var h_box_container_2: HBoxContainer = $ModsScroll/MarginContainer/ModsBox/HBoxContainer2

func _ready() -> void:
	h_box_container_2.visible = !use_file_browser;
	reload();

func reload() -> void:
	if DisplayServer.is_touchscreen_available():
		$ModsScroll.theme = load("res://mods-unpacked/CST1229-BALauncherModLoader/assets/big_scrollbar.tres");
	else:
		$ModsScroll.theme = null;
	
	if !mod_nodes.is_empty():
		save();
	for node in mod_nodes:
		node.queue_free();
	mod_nodes.clear();
		
	var mod_data := ModLoaderMod.get_mod_data_all();
	for mod_id in mod_data.keys():
		var data: ModData = mod_data[mod_id];
		var mod := MOD.instantiate();
		mod.mod_id = mod_id;
		mods_box.add_child(mod);
		
		mod_nodes.append(mod);
		mod.toggled.connect(func(_on: bool):
			if mod_main.mod_types[data] == "mods":
				warn_restart();
			mod.refresh_warnings();
			for other_mod: Node in mod_nodes:
				var other_data = (other_mod.mod_data as ModData);
				if other_data.manifest.incompatibilities.has(mod_id) or other_data.manifest.dependencies.has(mod_id):
					other_mod.refresh_warnings();
		);
		mod.config_changed.connect(func():
			changed_mod_config[mod_id] = true;
		);
	
	if mod_main.restart_warned:
		warn_restart();

func full_reload() -> void:
	click.play();
	$"/root/ModLoader/CST1229-BALauncherModLoader".reload_game_mods();
	reload();

func save() -> void:
	for mod_id in changed_mod_config.keys():
		var data := ModLoaderMod.get_mod_data(mod_id);
		if data.current_config:
			data.current_config.save_to_file();
		changed_mod_config.erase(mod_id);
	ModLoaderLog.info("Mod config saved", "CST1229-BALauncherModLoader");

func collapse_all() -> void:
	for mod_node in mod_nodes:
		if "expanded" in mod_node:
			mod_node.expanded = false;

func warn_restart() -> void:
	mod_main.restart_warned = true;
	restart_button.text = "(!) Restart Launcher";
	restart_button.add_theme_color_override(&"font_color", Color.YELLOW);
	restart_button.add_theme_color_override(&"font_hover_color", Color.YELLOW);
	restart_button.add_theme_color_override(&"font_focus_color", Color.YELLOW);
	restart_button.add_theme_color_override(&"font_pressed_color", Color.YELLOW);
	restart_button.add_theme_color_override(&"font_hover_pressed_color", Color.YELLOW);

func _on_restart_pressed() -> void:
	save();
	ModLoaderLog.info("Restarting game...", "CST1229-BALauncherModLoader");
	restart_imminent.emit();


func _on_open_mods_folder_pressed() -> void:
	click.play();
	OS.shell_show_in_file_manager("game-mods/", true);

func _on_tree_exiting() -> void:
	save();

func _on_open_save_folder_pressed() -> void:
	click.play();
	if !OS.has_feature("editor"):
		OS.shell_show_in_file_manager("mods/", true);
	else:
		OS.shell_show_in_file_manager("mods-unpacked/", true);

signal restart_imminent;
