extends VBoxContainer

var path: String:
	get: return owner.path;
	set(value): owner.path = value;

@onready var create_folder: Button = $Buttons/CreateFolder;
@onready var create_file: Button = $Buttons/CreateFile;
@onready var import_file: Button = $Buttons/ImportFile;
@onready var context_menu: Button = $Buttons/ContextMenu;
@onready var html_5_file_exchange: Node = $HTML5FileExchange;

@onready var list: ItemList = $ItemList;

func _ready() -> void:
	context_menu.disabled = list.is_anything_selected();
	import_file.visible = html_5_file_exchange.is_available();
	
	list.item_selected.connect(func(_index: int):
		# undo last_path file focusing hacks
		owner.last_path = path;
		context_menu.disabled = false;
	);
	list.item_activated.connect(func(index: int):
		var selectable := list.is_item_selectable(index);
		if !selectable:
			return;
		var text := list.get_item_text(index);
		if text.ends_with("/"):
			# undo last_path file focusing hacks
			owner.last_path = path;
		if text == "../":
			owner.go_back();
			return;
		if !owner.get_editor_for(path + text):
			OS.alert("This file can't be edited in the text editor (instead, export via the context menu and import via drag-and-drop or the 'Import Files' button).")
			return;
		path += text;
		owner.refresh();
	);
	list.item_clicked.connect(on_item_clicked);
	context_menu.pressed.connect(func():
		if list.is_anything_selected():
			on_item_clicked(list.get_selected_items()[0], context_menu.global_position, 2)
	);
	
	create_folder.pressed.connect(func():
		var _path := path;
		
		var edit := create_text_popup(func(new_text: String):
			if !new_text:
				return;
			if !new_text.is_valid_filename():
				OS.alert("Invalid filename.");
				return;
			var filepath := _path.path_join(new_text);
			var err := DirAccess.make_dir_absolute(filepath);
			if err:
				OS.alert(error_string(err));
				return;
			# focus the file
			owner.last_path = filepath;
			owner.refresh();
		);
		edit.text = "";
		edit.placeholder_text = "Folder name...";
		edit.grab_focus();
		edit.select_all();
	);
	create_file.pressed.connect(func():
		var _path := path;
		
		var edit := create_text_popup(func(new_text: String):
			if !new_text:
				return;
			if !new_text.is_valid_filename():
				OS.alert("Invalid filename.");
				return;
			var filepath := _path.path_join(new_text);
			if FileAccess.file_exists(filepath):
				OS.alert("File already exists.");
				return;
			var handle := FileAccess.open(filepath, FileAccess.WRITE);
			if !handle:
				OS.alert(error_string(FileAccess.get_open_error()));
				return;
			handle.close();
			# focus the file
			owner.last_path = filepath;
			owner.refresh();
		);
		edit.text = "";
		edit.placeholder_text = "File name...";
		edit.grab_focus();
		edit.select_all();
	);
	
	html_5_file_exchange.load_completed.connect(func(filename: String, bytes: PackedByteArray):
		if !visible: return;
		var filepath := path.path_join(filename);
		
		var handle := FileAccess.open(filepath, FileAccess.WRITE);
		if !handle:
			OS.alert(error_string(FileAccess.get_open_error()), "Couldn't write file");
			return;
		handle.store_buffer(bytes);
		handle.close();
		
		# focus the file
		owner.last_path = filepath;
		owner.refresh();
	);
	import_file.pressed.connect(html_5_file_exchange.load_file);

func on_item_clicked(index: int, at_position: Vector2, mouse_button_index: int):
	var selectable := list.is_item_selectable(index);
	if !selectable:
		return;
	if mouse_button_index != 2:
		return;
	var text := list.get_item_text(index);
	if text == "../":
		return;
	
	list.select(index);
	
	var menu := PopupMenu.new();
	menu.add_theme_font_override("font", load("res://mods-unpacked/CST1229-BALauncherModLoader/assets/nokiafc22.ttf"))
	var popup_scale := 1.0;
	if DisplayServer.is_touchscreen_available():
		popup_scale = 2.5;
	@warning_ignore("narrowing_conversion")
	menu.add_theme_font_size_override("font_size", 20 * popup_scale);
	
	menu.add_item("Rename", 1);
	if !text.ends_with("/"):
		menu.add_item("Duplicate", 5);
	if OS.has_feature("web") || OS.has_feature("mobile"):
		if text.ends_with("/"):
			menu.add_item("Export folder as .ZIP", 3);
		else:
			menu.add_item("Export file", 3);
	else:
		menu.add_item("Show in file manager", 4);
	menu.add_item("Delete", 2);
	
	var menu_pos := Vector2i(at_position) + Vector2i(list.global_position) + get_window().position;
	
	var full_path := get_full_path(text);
	menu.id_pressed.connect(func(id: int):
		match id:
			1:
				rename_file(text, menu_pos);
			2:
				delete_file(text);
			3:
				save_file(text);
			4:
				OS.shell_show_in_file_manager(ProjectSettings.globalize_path(full_path));
			5:
				duplicate_file(text);
	);
	
	list.add_child(menu);
	menu.popup(Rect2i(menu_pos, Vector2i.ZERO));
 
func refresh() -> void:
	list.clear();
	# scroll to top when going to a new folder
	if owner.last_path.ends_with("/"):
		list.get_v_scroll_bar().value = 0;
	
	var i = 0;
	if path != owner.ROOT_PATH:
		list.add_item("../", load("res://launcherassets/ui/icons/back.svg"));
		var back_path = path.path_join("..").simplify_path();
		if !back_path.ends_with("/"):
			back_path += "/";
		list.set_item_tooltip(i, "Go back to " + back_path);
		i += 1;
	
	var dir := DirAccess.open(path);
	if !dir:
		list.add_item("Could not open directory.", null, false);
		list.set_item_tooltip_enabled(i, false);
		i += 1;
		list.add_item(error_string(DirAccess.get_open_error()), null, false);
		list.set_item_tooltip_enabled(i, false);
		i += 1;
		return;
	
	var last_filename := "" if owner.last_path == "[keepscroll]" else (owner.last_path as String).trim_suffix("/").get_file();
	var focus_index := 0;
	
	var filenames := Array(dir.get_directories());
	filenames.sort_custom(func(a, b): return a.filenocasecmp_to(b) < 0);
	
	var folder_icon = load("res://launcherassets/ui/icons/folder.svg");
	var file_icon = load("res://launcherassets/ui/icons/file.svg");
	var barfy_icon = load("res://mods-unpacked/CST1229-BALauncherModLoader/assets/barfy.png");
	var launcher_icon = load("res://launcherassets/ui/pixelicons/folder.png");
	var editorsettings_icon = load("res://launcherassets/ui/pixelicons/editorsettings.png");
	var tools_icon = load("res://launcherassets/ui/pixelicons/tools.png");
	var bg_icon = load("res://launcherassets/ui/pixelicons/levelsettings/bg.png");
	var thumbnail_icon = load("res://launcherassets/ui/pixelicons/levelsettings/thumbnail.png");
	var info_icon = load("res://launcherassets/ui/pixelicons/levelsettings/info.png");
	
	var config_icon = load("res://launcherassets/ui/pixelicons/settings.png");
	var star_icon = load("res://launcherassets/ui/pixelicons/star.png");
	var save_icon = load("res://launcherassets/ui/pixelicons/save.png");
	var song_icon = load("res://launcherassets/ui/pixelicons/levelsettings/song.png");
	var game_icon = load("res://launcherassets/ui/iconnew.png");
	
	var separated := false;
	for filename in filenames:
		var icon = folder_icon;
		if path == owner.ROOT_PATH:
			match filename:
				"mods": icon = launcher_icon;
				"game-mods": icon = barfy_icon;
				"levels": icon = tools_icon;
				"structures": icon = bg_icon;
				"mod_configs": icon = editorsettings_icon;
				"pfps": icon = thumbnail_icon;
				"logs": icon = info_icon;
		
		list.add_item(filename + "/", icon);
		if last_filename == filename: focus_index = i;
		list.set_item_tooltip_enabled(i, false);
		i += 1;
		separated = true;
	
	filenames = Array(dir.get_files());
	filenames.sort_custom(func(a, b): return a.filenocasecmp_to(b) < 0);
	for filename in filenames:
		if separated:
			list.add_item("-------------------------------------------", null, false);
			list.set_item_tooltip_enabled(i, false);
			i += 1;
			separated = false;
			
		var icon = file_icon;
		if path == owner.ROOT_PATH:
			match filename:
				"settings.json": icon = config_icon;
				"favorites.save": icon = star_icon;
				"account.save", "dyes.save", "ghost.json", "hats.save", "levels.zip", "news.save", "settings.txt", "version.json", "level_pins.save", "version.txt", "story.txt":
					icon = save_icon;
				"game.pck": icon = game_icon;
				"songs.pck": icon = song_icon;
		
		list.add_item(filename, icon);
		if last_filename == filename: focus_index = i;
		list.set_item_tooltip_enabled(i, false);
		i += 1;
	list.select(focus_index);
	if owner.last_path != "[keepscroll]":
		list.ensure_current_is_visible();


func rename_file(text: String, at_position: Vector2i = get_tree().root.get_mouse_position()):
	var full_path := get_full_path(text);
	var filename := text.simplify_path();
	var edit := create_text_popup(func(new_text: String):
		if new_text == filename || !new_text:
			return;
		if !new_text.is_valid_filename():
			OS.alert("Invalid filename.");
			return;
		var fullfile := full_path.get_base_dir().path_join(new_text);
		if FileAccess.file_exists(fullfile):
			OS.alert("File already exists.");
			return;
		var err := DirAccess.rename_absolute(full_path, fullfile);
		if err:
			OS.alert(error_string(err));
			return;
		# focus the file
		owner.last_path = fullfile;
		owner.refresh();
	, at_position);
	edit.text = filename;
	edit.grab_focus();
	
	var ext := filename.get_extension();
	if ext != "":
		ext = "." + ext;
	edit.select(0, filename.length() - ext.length());

func duplicate_file(text: String, at_position: Vector2i = get_tree().root.get_mouse_position()):
	var full_path := get_full_path(text);
	var filename := text.simplify_path();
	
	var ext := filename.get_extension();
	if ext != "":
		ext = "." + ext;
		filename = filename.substr(0, filename.length() - ext.length()) + " copy" + ext;
	else:
		filename = filename + "copy";
	
	var edit := create_text_popup(func(new_text: String):
		if !new_text:
			return;
		if !new_text.is_valid_filename():
			OS.alert("Invalid filename.");
			return;
		var fullfile := full_path.get_base_dir().path_join(new_text);
		if FileAccess.file_exists(fullfile):
			OS.alert("File already exists!");
			return;
		var full_ogfile := full_path.get_base_dir().path_join(text);
		
		var bytes := FileAccess.get_file_as_bytes(full_ogfile);
		if FileAccess.get_open_error():
			OS.alert(error_string(FileAccess.get_open_error()), "Could not open original file");
			return;
		var handle := FileAccess.open(fullfile, FileAccess.WRITE);
		if FileAccess.get_open_error():
			OS.alert(error_string(FileAccess.get_open_error()), "Could not open new file");
			return;
		handle.store_buffer(bytes);
		handle.close();
		
		# focus the new file
		owner.last_path = fullfile;
		owner.refresh();
	, at_position);
	edit.text = filename;
	edit.placeholder_text = "File name...";
	edit.grab_focus();
	
	edit.select(0, filename.length() - ext.length());

func delete_file(text: String) -> void:
	var full_path := get_full_path(text);
	if OS.move_to_trash(ProjectSettings.globalize_path(full_path)):
		if text.ends_with("/"):
			delete_directory_recursive(full_path);
		else:
			DirAccess.remove_absolute(full_path);
	# keep scroll position
	owner.last_path = "[keepscroll]";
	owner.refresh();

func save_file(text: String) -> void:
	if OS.has_feature("android") && !("android.permission.MANAGE_EXTERNAL_STORAGE" in OS.get_granted_permissions()):
		OS.alert("Due to Android and Godot file sandboxing limitations (and my laziness), the game needs a 'manage every file' permission to have a file dialog for the downloads folder. You can disable this afterwards.");
		OS.request_permissions();
		return;
	
	var full_path := get_full_path(text);
	if text.ends_with("/"):
		full_path = "user://_folder_zip_deletable.zip";
		var packer := ZIPPacker.new();
		packer.open(full_path);
		zip_dir(packer, path.path_join(text), "");
		packer.close();
		
	var file := FileAccess.get_file_as_bytes(full_path);
	var export_filename := text;
	if text.ends_with("/"):
		DirAccess.remove_absolute(full_path);
		export_filename = text.trim_suffix("/") + ".zip";
	if OS.has_feature("android"):
		var dialog: FileDialog = load("res://mods-unpacked/CST1229-BALauncherModLoader/scenes/filebrowser/FileDialog.tscn").instantiate();
		dialog.current_dir = OS.get_system_dir(OS.SYSTEM_DIR_DOWNLOADS);
		dialog.current_file = export_filename;
		dialog.file_mode = FileDialog.FILE_MODE_SAVE_FILE;
		dialog.file_selected.connect(func(path: String):
			var handle := FileAccess.open(path, FileAccess.WRITE);
			if !handle:
				OS.alert(error_string(FileAccess.get_open_error()), "Could not open " + path);
			else:
				handle.store_buffer(file);
				handle.close();
				OS.alert("Saved file to:\n" + path);
		);
		add_child(dialog);
		dialog.visible = true;
		
		#var dir := OS.get_system_dir(OS.SYSTEM_DIR_DOWNLOADS);
		#var i := 2;
		#var old_export_filename := export_filename;
		#var ext := "." + export_filename.get_extension();
		#if ext == ".": ext = "";
		#var full_export_path := dir.path_join(export_filename);
		#
		#while FileAccess.file_exists(full_export_path):
			#export_filename = old_export_filename;
			#export_filename = export_filename.get_basename() + str(i) + ext;
			#full_export_path = dir.path_join(export_filename);
			#i += 1;
		#var handle := FileAccess.open(full_export_path, FileAccess.WRITE);
		#if !handle:
			#OS.alert(error_string(FileAccess.get_open_error()), "Could not open " + full_export_path);
		#else:
			#handle.store_buffer(file);
			#handle.close();
			#OS.alert("Saved file to:\n" + full_export_path);
	else:
		JavaScriptBridge.download_buffer(file, export_filename);

func zip_dir(packer: ZIPPacker, real_path: String, inside_path: String) -> void:
	for file in DirAccess.get_files_at(real_path):
		packer.start_file(inside_path + file);
		packer.write_file(FileAccess.get_file_as_bytes(real_path + file));
		packer.close_file();
	for dir in DirAccess.get_directories_at(real_path):
		zip_dir(packer, real_path + dir + "/", inside_path + dir + "/");


func get_full_path(text: String) -> String:
	return path.path_join(text).simplify_path();

func delete_directory_recursive(at_path: String) -> void:
	for file in DirAccess.get_files_at(at_path):
		DirAccess.remove_absolute(at_path.path_join(file));
	for dir in DirAccess.get_directories_at(at_path):
		delete_directory_recursive(at_path.path_join(dir));
	DirAccess.remove_absolute(at_path);

func create_text_popup(on_submit: Callable, at_position: Vector2i = get_tree().root.get_mouse_position()) -> LineEdit:
	var popup := PopupPanel.new();
	popup.add_theme_stylebox_override("panel", StyleBoxEmpty.new());
	var popup_scale := 1.0;
	if DisplayServer.is_touchscreen_available():
		popup_scale = 1.5;
	
	var edit := LineEdit.new();
	edit.add_theme_font_override("font", load("res://mods-unpacked/CST1229-BALauncherModLoader/assets/nokiafc22.ttf"));
	@warning_ignore("narrowing_conversion")
	edit.add_theme_font_size_override("font_size", 16 * popup_scale);
	edit.text_submitted.connect(func(new_text: String):
		if edit.has_meta("no_more_submits"):
			return;
		edit.set_meta("no_more_submits", true);
		popup.queue_free();
		edit.queue_free();
		on_submit.call(new_text);
	);
	edit.tree_exiting.connect(func():
		if edit.has_meta("no_more_submits"):
			return;
		edit.text_submitted.emit(edit.text);
	);
	edit.gui_input.connect(func(ev: InputEvent):
		if ev is InputEventKey:
			var evk := ev as InputEventKey;
			if ev.is_pressed() && !ev.is_echo() && evk.physical_keycode == KEY_ESCAPE:
				edit.set_meta("no_more_submits", true);
				popup.queue_free();
				edit.queue_free();
	);
	popup.popup_hide.connect(popup.queue_free);
	popup.add_child(edit);
	
	list.add_child(popup);
	@warning_ignore("narrowing_conversion")
	popup.popup(Rect2i(at_position, Vector2i(384 * popup_scale, 32 * popup_scale)));
	return edit;

func after_setup():
	list.grab_focus();

func _unhandled_input(ev: InputEvent):
	if ev is InputEventKey:
		var kev := ev as InputEventKey;
		if kev.is_pressed() && !kev.is_echo():
			var selected_index := -1;
			var selected_text := "";
			var items := list.get_selected_items();
			if !items.is_empty():
				var index := items[0];
				var selectable := list.is_item_selectable(index);
				if selectable:
					var text := list.get_item_text(index);
					if text != "../":
						selected_index = index;
						selected_text = text;
			
			if kev.physical_keycode == KEY_BACKSPACE && path != owner.ROOT_PATH:
				owner.go_back();
			elif kev.physical_keycode == KEY_DELETE && selected_index >= 0:
				delete_file(selected_text);
			elif kev.keycode == KEY_D && selected_index >= 0 && kev.ctrl_pressed:
				duplicate_file(selected_text);
			elif kev.physical_keycode == KEY_F2 && selected_index >= 0:
				rename_file(selected_text);

func _enter_tree() -> void:
	get_tree().root.get_window().files_dropped.connect(on_files_dropped);
	get_window().files_dropped.connect(on_files_dropped);

func on_files_dropped(files: PackedStringArray):
	if !visible:
		return;
	for file in files:
		DirAccess.copy_absolute(file, path.path_join(file.get_file()));
	refresh();
	
func _exit_tree() -> void:
	get_tree().root.get_window().get_window().files_dropped.disconnect(
		on_files_dropped
	);
	get_window().files_dropped.disconnect(on_files_dropped);
